<?php
/**
 * Plugin Name: WooCommerce Conversion Tracker
 * Plugin URI: https://kureci.dev
 * Description: WooCommerce olaylarını Google Ads ve Facebook Pixel'e gönderir (dataLayer'a bağımlı değildir).
 * Version: 1.2.0
 * Author: Creanify
 * Author URI: https://kureci.dev
 * Text Domain: wc-conversion-tracker
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.5
 * WC requires at least: 3.0
 * WC tested up to: 8.9
 */

if (!defined('ABSPATH')) { exit; }

// WooCommerce kontrol
add_action('admin_init', function() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            echo '<div class="notice notice-error"><p>';
            echo esc_html__('WooCommerce Conversion Tracker requires WooCommerce to be installed and active.', 'wc-conversion-tracker');
            echo '</p></div>';
        });
        deactivate_plugins(plugin_basename(__FILE__));
    }
});

// Sabitler
define('WCT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WCT_PLUGIN_PATH', plugin_dir_path(__FILE__));

class WC_Conversion_Tracker {

    public function __construct() {
        add_action('init', [$this, 'init']);
    }

    public function init() {
        // Admin
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);

        // Global tracking (head)
        add_action('wp_head', [$this, 'add_global_tracking_code'], 20);

        // Hard-mode tetikleyiciler (footer)
        add_action('wp_footer', [$this, 'render_product_view'], 5);
        add_action('wp_footer', [$this, 'render_cart_view'], 5);
        add_action('wp_footer', [$this, 'render_add_to_cart_listener'], 5);
        add_action('wp_footer', [$this, 'render_purchase_on_thankyou'], 5);

        // (Opsiyonel) dataLayer listener da kalsın
        add_action('wp_footer', [$this, 'add_datalayer_listener'], 20);
    }

    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Conversion Tracking', 'wc-conversion-tracker'),
            __('Conversion Tracking', 'wc-conversion-tracker'),
            'manage_woocommerce',
            'wc-conversion-tracking',
            [$this, 'admin_page']
        );
    }

    public function register_settings() {
        // Google Ads
        register_setting('wct_settings', 'wct_enable_google_ads', ['type' => 'integer', 'sanitize_callback' => 'absint']);
        register_setting('wct_settings', 'wct_google_ads_id', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('wct_settings', 'wct_google_ads_label_addtocart', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('wct_settings', 'wct_google_ads_label_purchase', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('wct_settings', 'wct_google_ads_label_viewcart', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);
        register_setting('wct_settings', 'wct_google_ads_label_pageview', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);

        // GA4
        register_setting('wct_settings', 'wct_enable_ga', ['type' => 'integer', 'sanitize_callback' => 'absint']);
        register_setting('wct_settings', 'wct_ga_measurement_id', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);

        // Facebook
        register_setting('wct_settings', 'wct_enable_facebook', ['type' => 'integer', 'sanitize_callback' => 'absint']);
        register_setting('wct_settings', 'wct_facebook_pixel_id', ['type' => 'string', 'sanitize_callback' => 'sanitize_text_field']);
    }

    public function admin_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div class="notice notice-info">
                <p><?php _e('Bu eklenti WooCommerce olaylarını doğrudan izler, dataLayer yoksa bile dönüşüm yollar.', 'wc-conversion-tracker'); ?></p>
                <p><strong><?php _e('Geliştirici:', 'wc-conversion-tracker'); ?></strong> <a href="https://kureci.dev" target="_blank" rel="noopener">Kureci DEV</a></p>
            </div>

            <div id="wct-tabs">
                <ul class="nav-tab-wrapper">
                    <li><a href="#tab-google-ads" class="nav-tab nav-tab-active"><?php _e('Google Ads & GA4', 'wc-conversion-tracker'); ?></a></li>
                    <li><a href="#tab-facebook" class="nav-tab"><?php _e('Facebook Pixel', 'wc-conversion-tracker'); ?></a></li>
                    <li><a href="#tab-setup" class="nav-tab"><?php _e('Kurulum Rehberi', 'wc-conversion-tracker'); ?></a></li>
                </ul>

                <form method="post" action="options.php">
                    <?php settings_fields('wct_settings'); ?>

                    <!-- Google Ads & GA4 Tab -->
                    <div id="tab-google-ads" class="tab-content">
                        <table class="form-table">
                            <tbody>
                                <tr>
                                    <th scope="row"><label for="wct_enable_google_ads"><?php _e('Google Ads Aktif', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="checkbox" id="wct_enable_google_ads" name="wct_enable_google_ads" value="1" <?php checked(1, get_option('wct_enable_google_ads'), true); ?> />
                                        <p class="description"><?php _e('Google Ads dönüşüm takibini etkinleştir', 'wc-conversion-tracker'); ?></p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_google_ads_id"><?php _e('Google Ads Conversion ID', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="text" id="wct_google_ads_id" name="wct_google_ads_id"
                                               value="<?php echo esc_attr(get_option('wct_google_ads_id')); ?>"
                                               placeholder="AW-XXXXXXXXX" class="regular-text" />
                                        <p class="description"><?php _e('AW ile başlayan Conversion ID (örn: AW-123456789).', 'wc-conversion-tracker'); ?></p>
                                    </td>
                                </tr>

                                <tr>
                                    <th colspan="2"><h3><?php _e('GA4 Ayarları', 'wc-conversion-tracker'); ?></h3></th>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_enable_ga"><?php _e('GA4 Aktif', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="checkbox" id="wct_enable_ga" name="wct_enable_ga" value="1" <?php checked(1, get_option('wct_enable_ga'), true); ?> />
                                        <p class="description"><?php _e('Google Analytics 4 ölçümü etkin olsun.', 'wc-conversion-tracker'); ?></p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_ga_measurement_id"><?php _e('GA4 Measurement ID', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="text" id="wct_ga_measurement_id" name="wct_ga_measurement_id"
                                               value="<?php echo esc_attr(get_option('wct_ga_measurement_id')); ?>"
                                               placeholder="G-XXXXXXXXXX" class="regular-text" />
                                        <p class="description"><?php _e('G ile başlayan GA4 Measurement ID (örn: G-ABCDEFGH12).', 'wc-conversion-tracker'); ?></p>
                                    </td>
                                </tr>

                                <tr>
                                    <th colspan="2"><h3><?php _e('Google Ads Dönüşüm Etiketleri', 'wc-conversion-tracker'); ?></h3></th>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_google_ads_label_addtocart"><?php _e('Add to Cart Label', 'wc-conversion-tracker'); ?></label></th>
                                    <td><input type="text" id="wct_google_ads_label_addtocart" name="wct_google_ads_label_addtocart" value="<?php echo esc_attr(get_option('wct_google_ads_label_addtocart')); ?>" class="regular-text" /></td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_google_ads_label_viewcart"><?php _e('View Cart Label', 'wc-conversion-tracker'); ?></label></th>
                                    <td><input type="text" id="wct_google_ads_label_viewcart" name="wct_google_ads_label_viewcart" value="<?php echo esc_attr(get_option('wct_google_ads_label_viewcart')); ?>" class="regular-text" /></td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_google_ads_label_purchase"><?php _e('Purchase Label', 'wc-conversion-tracker'); ?></label></th>
                                    <td><input type="text" id="wct_google_ads_label_purchase" name="wct_google_ads_label_purchase" value="<?php echo esc_attr(get_option('wct_google_ads_label_purchase')); ?>" class="regular-text" /></td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_google_ads_label_pageview"><?php _e('Page View Label', 'wc-conversion-tracker'); ?></label></th>
                                    <td><input type="text" id="wct_google_ads_label_pageview" name="wct_google_ads_label_pageview" value="<?php echo esc_attr(get_option('wct_google_ads_label_pageview')); ?>" class="regular-text" /></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <!-- Facebook Tab -->
                    <div id="tab-facebook" class="tab-content" style="display:none;">
                        <table class="form-table">
                            <tbody>
                                <tr>
                                    <th scope="row"><label for="wct_enable_facebook"><?php _e('Facebook Pixel Aktif', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="checkbox" id="wct_enable_facebook" name="wct_enable_facebook" value="1" <?php checked(1, get_option('wct_enable_facebook'), true); ?> />
                                        <p class="description"><?php _e('Facebook Pixel dönüşüm takibini etkinleştir', 'wc-conversion-tracker'); ?></p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="wct_facebook_pixel_id"><?php _e('Facebook Pixel ID', 'wc-conversion-tracker'); ?></label></th>
                                    <td>
                                        <input type="text" id="wct_facebook_pixel_id" name="wct_facebook_pixel_id"
                                               value="<?php echo esc_attr(get_option('wct_facebook_pixel_id')); ?>"
                                               placeholder="XXXXXXXXXXXXXXX" class="regular-text" />
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>

                    <!-- Setup Guide Tab -->
                    <div id="tab-setup" class="tab-content" style="display:none;">
                        <div class="card">
                            <h2><?php _e('Kurulum Rehberi', 'wc-conversion-tracker'); ?></h2>
                            <ol>
                                <li><?php _e('AW- (Ads) ve G- (GA4) ID’lerini girip ilgili anahtarları etkinleştirin.', 'wc-conversion-tracker'); ?></li>
                                <li><?php _e('LiteSpeed Cache: /odeme/order-received/* cache dışı, gtag dosyaları JS exclude.', 'wc-conversion-tracker'); ?></li>
                                <li><?php _e('Tarayıcı konsolunda “WCT:” loglarını kontrol edin.', 'wc-conversion-tracker'); ?></li>
                            </ol>
                        </div>
                    </div>

                    <?php submit_button(); ?>
                </form>
            </div>
        </div>

        <style>
        .nav-tab-wrapper { border-bottom:1px solid #ccc; margin:20px 0; }
        .tab-content { background:#fff; border:1px solid #ccd0d4; border-top:none; padding:20px; }
        .card { background:#fff; border:1px solid #ccd0d4; border-radius:4px; padding:20px; margin-top:20px; }
        .card h2, .card h3 { margin-top:0; }
        </style>

        <script>
        jQuery(document).ready(function($){
            $('.nav-tab').on('click', function(e){
                e.preventDefault();
                var target = $(this).attr('href');
                $('.nav-tab').removeClass('nav-tab-active');
                $(this).addClass('nav-tab-active');
                $('.tab-content').hide();
                $(target).show();
            });
        });
        </script>
        <?php
    }

    // Head: GA4 (loader) + Ads (config) + FB Pixel
    public function add_global_tracking_code() {
        $ads_id      = trim((string) get_option('wct_google_ads_id'));          // AW-XXXXXXX
        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ga_id       = trim((string) get_option('wct_ga_measurement_id'));       // G-XXXXXXX
        $ga_enabled  = (int) get_option('wct_enable_ga') === 1;

        $loader_id = ($ga_enabled && $ga_id) ? $ga_id : ( ($ads_enabled && $ads_id) ? $ads_id : '' );

        if ($loader_id) : ?>
            <!-- Global Site Tag (gtag.js) - WCT -->
            <script async src="https://www.googletagmanager.com/gtag/js?id=<?php echo esc_js($loader_id); ?>"></script>
            <script>
              window.dataLayer = window.dataLayer || [];
              function gtag(){dataLayer.push(arguments);}
              gtag('js', new Date());

              <?php if ($ga_enabled && $ga_id): ?>
              gtag('config', '<?php echo esc_js($ga_id); ?>');
              <?php endif; ?>

              <?php if ($ads_enabled && $ads_id): ?>
              gtag('config', '<?php echo esc_js($ads_id); ?>');
              <?php endif; ?>
            </script>
        <?php endif;

        // Facebook Pixel
        $fb_id      = trim((string) get_option('wct_facebook_pixel_id'));
        $fb_enabled = (int) get_option('wct_enable_facebook') === 1;

        if ($fb_enabled && $fb_id) { ?>
            <!-- Facebook Pixel Code - WCT -->
            <script>
              !function(f,b,e,v,n,t,s)
              {if(f.fbq)return;n=f.fbq=function(){n.callMethod?
              n.callMethod.apply(n,arguments):n.queue.push(arguments)};
              if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';
              n.queue=[];t=b.createElement(e);t.async=!0;
              t.src=v;s=b.getElementsByTagName(e)[0];
              s.parentNode.insertBefore(t,s)}(window, document,'script',
              'https://connect.facebook.net/en_US/fbevents.js');
              fbq('init', '<?php echo esc_js($fb_id); ?>');
              fbq('track', 'PageView');
            </script>
            <noscript><img height="1" width="1" style="display:none"
              src="https://www.facebook.com/tr?id=<?php echo esc_attr($fb_id); ?>&ev=PageView&noscript=1" /></noscript>
        <?php }
    }

    /*** HARD-MODE TETIKLEYICILER ***/

    // Ürün sayfası: AW PageView
    public function render_product_view() {
        if (!function_exists('is_product') || !is_product()) return;

        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ads_id      = trim((string) get_option('wct_google_ads_id'));
        $label       = trim((string) get_option('wct_google_ads_label_pageview'));

        if ($ads_enabled && $ads_id && $label) : ?>
        <script>
        (function(){
            if (typeof gtag === 'undefined') return;
            console.log('WCT: AW page_view (product)');
            gtag('event', 'conversion', {
                'send_to': '<?php echo esc_js($ads_id); ?>/<?php echo esc_js($label); ?>',
                'value': 0,
                'currency': 'TRY'
            });
        })();
        </script>
        <?php endif;
    }

    // Sepet sayfası: AW ViewCart
    public function render_cart_view() {
        if (!function_exists('is_cart') || !is_cart()) return;

        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ads_id      = trim((string) get_option('wct_google_ads_id'));
        $label       = trim((string) get_option('wct_google_ads_label_viewcart'));

        if (!$ads_enabled || !$ads_id || !$label) return;

        $cart_total = 0;
        if (function_exists('WC') && WC()->cart) {
            $cart_total = (float) WC()->cart->get_total('edit'); // vergisiz toplam (metin değil sayı)
        }
        ?>
        <script>
        (function(){
            if (typeof gtag === 'undefined') return;
            console.log('WCT: AW view_cart');
            gtag('event', 'conversion', {
                'send_to': '<?php echo esc_js($ads_id); ?>/<?php echo esc_js($label); ?>',
                'value': <?php echo json_encode( (float) $cart_total ); ?>,
                'currency': '<?php echo esc_js( get_woocommerce_currency() ); ?>'
            });
        })();
        </script>
        <?php
    }

    // Sepete ekle: WooCommerce JS event dinleyicisi
    public function render_add_to_cart_listener() {
        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ads_id      = trim((string) get_option('wct_google_ads_id'));
        $label       = trim((string) get_option('wct_google_ads_label_addtocart'));
        $fb_enabled  = (int) get_option('wct_enable_facebook') === 1;

        if (!$ads_enabled && !$fb_enabled) return;
        ?>
        <script>
        (function($){
            if (typeof jQuery === 'undefined') return;
            // WooCommerce, ürün sepete eklendiğinde body’ye 'added_to_cart' event'i atar
            jQuery(document.body).on('added_to_cart', function(event, fragments, cart_hash, $button){
                try {
                    var value = 0;
                    var currency = '<?php echo esc_js( get_woocommerce_currency() ); ?>';
                    <?php if ($ads_enabled && $ads_id && $label): ?>
                    if (typeof gtag !== 'undefined') {
                        console.log('WCT: AW add_to_cart');
                        gtag('event', 'conversion', {
                            'send_to': '<?php echo esc_js($ads_id); ?>/<?php echo esc_js($label); ?>',
                            'value': value,
                            'currency': currency
                        });
                    }
                    <?php endif; ?>
                    <?php if ($fb_enabled): ?>
                    if (typeof fbq !== 'undefined') {
                        console.log('WCT: FB AddToCart');
                        fbq('track', 'AddToCart', {
                            value: value,
                            currency: currency
                        });
                    }
                    <?php endif; ?>
                } catch(e){ console.warn('WCT add_to_cart error', e); }
            });
        })(jQuery);
        </script>
        <?php
    }

    // Teşekkür sayfası: Purchase (AW & FB) – dataLayer’sız garanti tetikleme
    public function render_purchase_on_thankyou() {
        if (!function_exists('is_order_received_page') || !is_order_received_page()) return;

        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ads_id      = trim((string) get_option('wct_google_ads_id'));
        $label       = trim((string) get_option('wct_google_ads_label_purchase'));
        $fb_enabled  = (int) get_option('wct_enable_facebook') === 1;

        // Order ID bul
        $order_id = 0;
        if (isset($_GET['key'])) {
            $order_id = wc_get_order_id_by_order_key(wp_unslash($_GET['key']));
        }
        if (!$order_id && isset($_GET['order'])) {
            $order_id = absint($_GET['order']);
        }
        if (!$order_id) return;

        $order = wc_get_order($order_id);
        if (!$order) return;

        $value       = (float) $order->get_total();
        $currency    = (string) $order->get_currency();
        $transaction = (string) $order->get_order_key();

        // FB için içerik listesi (id’ler)
        $content_ids = [];
        foreach ($order->get_items() as $item) {
            $pid = $item->get_product_id();
            if ($pid) { $content_ids[] = (string) $pid; }
        }
        ?>
        <script>
        (function(){
            var value    = <?php echo json_encode($value); ?>;
            var currency = <?php echo json_encode($currency); ?>;
            var txid     = <?php echo json_encode($transaction); ?>;
            <?php if ($ads_enabled && $ads_id && $label): ?>
            if (typeof gtag !== 'undefined') {
                console.log('WCT: AW purchase');
                gtag('event', 'conversion', {
                    'send_to': '<?php echo esc_js($ads_id); ?>/<?php echo esc_js($label); ?>',
                    'value': value,
                    'currency': currency,
                    'transaction_id': txid
                });
            }
            <?php endif; ?>

            <?php if ($fb_enabled): ?>
            if (typeof fbq !== 'undefined') {
                console.log('WCT: FB Purchase');
                fbq('track', 'Purchase', {
                    value: value,
                    currency: currency,
                    content_ids: <?php echo wp_json_encode($content_ids); ?>,
                    content_type: 'product'
                });
            }
            <?php endif; ?>
        })();
        </script>
        <?php
    }

    /*** (Opsiyonel) dataLayer listener – kalabilir ***/
    public function add_datalayer_listener() {
        $ads_enabled = (int) get_option('wct_enable_google_ads') === 1;
        $ads_id      = trim((string) get_option('wct_google_ads_id'));
        $fb_enabled  = (int) get_option('wct_enable_facebook') === 1;
        $fb_id       = trim((string) get_option('wct_facebook_pixel_id'));

        if (!$ads_enabled && !$fb_enabled) { return; } ?>
        <script>
        (function() {
            console.log('WCT: DataLayer listener (optional) ready');
            window.dataLayer = window.dataLayer || [];
            if (!window.originalDataLayerPush) {
                window.originalDataLayerPush = window.dataLayer.push;
            }
            window.dataLayer.push = function() {
                window.originalDataLayerPush.apply(window.dataLayer, arguments);
                var ev = arguments[0];
                if (ev && ev.event) console.log('WCT: dataLayer event', ev.event, ev);
            };
        })();
        </script>
        <?php
    }
}

// Başlat
new WC_Conversion_Tracker();

// Aktivasyon varsayılanları
register_activation_hook(__FILE__, function() {
    add_option('wct_enable_google_ads', 0);
    add_option('wct_enable_facebook', 0);
    add_option('wct_enable_ga', 0);
});

// Deaktivasyon
register_deactivation_hook(__FILE__, function() {
    // Cleanup gerekirse
});