<?php
/**
 * Plugin Name: Yapay zeka için SEO - Zohi
 * Description: SEO verilerini JSON & XML olarak dışa aktarır ve REST API ile paylaşır. Bu sürüm uyumluluk için sadeleştirilmiştir (opsiyonel basit token kontrolü).
 * Version: 1.3.0
 * Author: Zohi
 * Author URI: https://www.zohi.net
 * Plugin URI: https://www.zohi.net
 * Text Domain: zohi-ai-seo
 * Requires at least: 5.2
 * Requires PHP: 7.0
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

if ( ! class_exists( 'Zohi_AI_SEO' ) ) :

class Zohi_AI_SEO {
    const OPTION_KEY    = 'zohi_ai_seo_options';
    const CRON_HOOK     = 'zohi_ai_seo_build_exports';
    const UPLOAD_SUBDIR = 'zohi-seo';
    const VERSION       = '1.3.0';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'rest_api_init', array( $this, 'register_rest' ) );
        add_action( self::CRON_HOOK, array( $this, 'build_exports' ) );
        add_filter( 'cron_schedules', array( $this, 'add_custom_schedules' ) );

        register_activation_hook( __FILE__, array( 'Zohi_AI_SEO', 'activate' ) );
        register_deactivation_hook( __FILE__, array( 'Zohi_AI_SEO', 'deactivate' ) );
    }

    public static function activate() {
        $defaults = array(
            'frequency'         => 'hourly',
            'post_types'        => array( 'post', 'page' ),
            'include_tax'       => true,
            'detect_seo_plugins'=> true,
            'try_fetch_sitemap' => true,
            // Basit güvenlik (opsiyonel)
            'require_token'     => false,
            'api_token'         => '',
        );
        add_option( self::OPTION_KEY, $defaults );

        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            wp_schedule_event( time() + 60, 'hourly', self::CRON_HOOK );
        }
    }

    public static function deactivate() {
        $timestamp = wp_next_scheduled( self::CRON_HOOK );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, self::CRON_HOOK );
        }
    }

    public function add_custom_schedules( $schedules ) {
        if ( ! isset( $schedules['weekly'] ) ) {
            $schedules['weekly'] = array( 'interval' => WEEK_IN_SECONDS, 'display' => __( 'Weekly', 'zohi-ai-seo' ) );
        }
        if ( ! isset( $schedules['monthly'] ) ) {
            $schedules['monthly'] = array( 'interval' => 30 * DAY_IN_SECONDS, 'display' => __( 'Monthly', 'zohi-ai-seo' ) );
        }
        return $schedules;
    }

    public function add_admin_page() {
        add_options_page( __( 'Yapay zeka için SEO - Zohi', 'zohi-ai-seo' ), __( 'AI için SEO', 'zohi-ai-seo' ), 'manage_options', 'zohi-ai-seo', array( $this, 'render_settings_page' ) );
    }

    public function register_settings() {
        register_setting( self::OPTION_KEY, self::OPTION_KEY, array( $this, 'sanitize_options' ) );

        add_settings_section( 'zohi_ai_seo_main', __( 'Genel Ayarlar', 'zohi-ai-seo' ), array( $this, 'section_desc' ), 'zohi-ai-seo' );

        add_settings_field( 'frequency', __( 'Veri toplama sıklığı', 'zohi-ai-seo' ), array( $this, 'field_frequency' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
        add_settings_field( 'post_types', __( 'Dahil edilecek içerik tipleri', 'zohi-ai-seo' ), array( $this, 'field_post_types' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
        add_settings_field( 'include_tax', __( 'Kategoriler & Etiketler dahil', 'zohi-ai-seo' ), array( $this, 'field_include_tax' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
        add_settings_field( 'detect_seo_plugins', __( 'SEO eklentilerini tespit', 'zohi-ai-seo' ), array( $this, 'field_detect_seo_plugins' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
        add_settings_field( 'try_fetch_sitemap', __( 'Sitemap bağlantıları', 'zohi-ai-seo' ), array( $this, 'field_try_fetch_sitemap' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );

        // Basit token (opsiyonel, sadece query param ile)
        add_settings_field( 'require_token', __( 'REST için token zorunlu', 'zohi-ai-seo' ), array( $this, 'field_require_token' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
        add_settings_field( 'api_token', __( 'Gizli anahtar (token)', 'zohi-ai-seo' ), array( $this, 'field_api_token' ), 'zohi-ai-seo', 'zohi_ai_seo_main' );
    }

    public function section_desc() {
        echo '<p>' . esc_html__( 'SEO verileri JSON & XML olarak oluşturulur ve REST API ile paylaşılır. Bu sürüm uyumluluk için basitleştirilmiştir.', 'zohi-ai-seo' ) . '</p>';
    }

    public function sanitize_options( $opts ) {
        $clean = array();
        $allowed_freq = array( 'hourly', 'daily', 'weekly', 'monthly' );
        $clean['frequency']   = in_array( isset($opts['frequency']) ? $opts['frequency'] : 'hourly', $allowed_freq, true ) ? $opts['frequency'] : 'hourly';
        $pt = isset($opts['post_types']) ? (array)$opts['post_types'] : array( 'post', 'page' );
        $clean['post_types']  = array();
        foreach ( $pt as $p ) { $clean['post_types'][] = sanitize_text_field( $p ); }
        $clean['include_tax'] = ! empty( $opts['include_tax'] );
        $clean['detect_seo_plugins'] = ! empty( $opts['detect_seo_plugins'] );
        $clean['try_fetch_sitemap']  = ! empty( $opts['try_fetch_sitemap'] );

        $clean['require_token'] = ! empty( $opts['require_token'] );
        $token = isset($opts['api_token']) ? $opts['api_token'] : '';
        $clean['api_token'] = sanitize_text_field( $token );
        if ( $clean['require_token'] && $clean['api_token'] === '' ) {
            $clean['api_token'] = wp_generate_password( 32, false, false );
        }

        $current = get_option( self::OPTION_KEY );
        if ( $current && isset( $current['frequency'] ) && $current['frequency'] !== $clean['frequency'] ) {
            $this->reschedule( $clean['frequency'] );
        }
        return $clean;
    }

    private function reschedule( $freq ) {
        $timestamp = wp_next_scheduled( self::CRON_HOOK );
        if ( $timestamp ) wp_unschedule_event( $timestamp, self::CRON_HOOK );
        wp_schedule_event( time() + 60, $freq, self::CRON_HOOK );
    }

    public function field_frequency() {
        $opts = get_option( self::OPTION_KEY );
        $value = isset($opts['frequency']) ? $opts['frequency'] : 'hourly';
        $choices = array( 'hourly'=>__( 'Her saat başı','zohi-ai-seo' ), 'daily'=>__( 'Günlük','zohi-ai-seo' ), 'weekly'=>__( 'Haftalık','zohi-ai-seo' ), 'monthly'=>__( 'Aylık','zohi-ai-seo' ) );
        echo '<select name="'.esc_attr(self::OPTION_KEY).'[frequency]">';
        foreach ( $choices as $k=>$label ) printf('<option value="%s"%s>%s</option>', esc_attr($k), selected($value,$k,false), esc_html($label));
        echo '</select>';
    }

    public function field_post_types() {
        $opts = get_option( self::OPTION_KEY );
        $selected = isset($opts['post_types']) ? (array)$opts['post_types'] : array('post','page');
        $pts = get_post_types( array( 'public'=>true ), 'objects' );
        foreach ( $pts as $pt=>$obj ) {
            printf('<label style="display:block"><input type="checkbox" name="%s[post_types][]" value="%s" %s> %s</label>', esc_attr(self::OPTION_KEY), esc_attr($pt), checked(in_array($pt,$selected,true), true, false), esc_html($obj->labels->name));
        }
    }

    public function field_include_tax() { $opts=get_option(self::OPTION_KEY); printf('<label><input type="checkbox" name="%s[include_tax]" value="1" %s> %s</label>', esc_attr(self::OPTION_KEY), checked(!empty($opts['include_tax']), true, false), esc_html__('Kategoriler/Etiketler eklenecek','zohi-ai-seo')); }
    public function field_detect_seo_plugins() { $opts=get_option(self::OPTION_KEY); printf('<label><input type="checkbox" name="%s[detect_seo_plugins]" value="1" %s> %s</label>', esc_attr(self::OPTION_KEY), checked(!empty($opts['detect_seo_plugins']), true, false), esc_html__('Yoast/RankMath/AIOSEO tespiti','zohi-ai-seo')); }
    public function field_try_fetch_sitemap() { $opts=get_option(self::OPTION_KEY); printf('<label><input type="checkbox" name="%s[try_fetch_sitemap]" value="1" %s> %s</label>', esc_attr(self::OPTION_KEY), checked(!empty($opts['try_fetch_sitemap']), true, false), esc_html__('Yaygın sitemap yollarını listele','zohi-ai-seo')); }

    public function field_require_token() { $opts=get_option(self::OPTION_KEY); printf('<label><input type="checkbox" name="%s[require_token]" value="1" %s> %s</label>', esc_attr(self::OPTION_KEY), checked(!empty($opts['require_token']), true, false), esc_html__('REST için token zorunlu olsun (sadece ?token=)','zohi-ai-seo')); }
    public function field_api_token() {
        $opts=get_option(self::OPTION_KEY);
        $token = isset($opts['api_token']) ? $opts['api_token'] : '';
        if ( empty($token) ) { $token = wp_generate_password(32,false,false); $opts['api_token']=$token; update_option(self::OPTION_KEY,$opts); }
        echo '<input type="text" readonly value="'.esc_attr($token).'" style="width:420px" />';
        echo '<p class="description">'.esc_html__('Kullanım: /wp-json/zohi-seo/v1/export?token=BURAYA_TOKEN','zohi-ai-seo').'</p>';
    }

    public function render_settings_page() {
        if ( isset($_POST['zohi_ai_seo_run_now']) && check_admin_referer('zohi_ai_seo_run_now') ) { $this->build_exports(); echo '<div class="updated"><p>'.esc_html__('Dışa aktarma çalıştırıldı.','zohi-ai-seo').'</p></div>'; }
        echo '<div class="wrap"><h1>'.esc_html__('Yapay zeka için SEO - Zohi','zohi-ai-seo').'</h1><p>'.esc_html__('Basit ve uyumlu sürüm.','zohi-ai-seo').'</p>';
        echo '<form method="post" action="options.php">'; settings_fields(self::OPTION_KEY); do_settings_sections('zohi-ai-seo'); submit_button(); echo '</form><hr/>';
        echo '<form method="post">'; wp_nonce_field('zohi_ai_seo_run_now'); submit_button(__('Şimdi dışa aktarmayı çalıştır','zohi-ai-seo'),'secondary','zohi_ai_seo_run_now'); echo '</form>';
        $paths=$this->get_export_paths(); echo '<p><strong>'.esc_html__('Son çıktılar:','zohi-ai-seo').'</strong></p><ul>'; if(file_exists($paths['json'])) echo '<li>JSON: <code>'.esc_html($paths['json_url']).'</code></li>'; if(file_exists($paths['xml'])) echo '<li>XML: <code>'.esc_html($paths['xml_url']).'</code></li>'; echo '</ul></div>';
    }

    public function register_rest() {
        register_rest_route('zohi-seo/v1','/export', array(
            'methods'=>'GET',
            'callback'=>array($this,'rest_export'),
            'permission_callback'=>array($this,'rest_permission'),
        ));
    }

    public function rest_permission( $request ) {
        $opts = get_option(self::OPTION_KEY);
        if ( empty($opts['require_token']) ) return true;
        $expected = isset($opts['api_token']) ? $opts['api_token'] : '';
        $provided = (string)$request->get_param('token');
        if ( $expected !== '' && $provided !== '' && hash_equals($expected, $provided) ) return true;
        return new WP_Error('zohi_seo_unauthorized', __('Geçersiz veya eksik token.','zohi-ai-seo'), array('status'=>401));
    }

    public function rest_export( $request ) {
        $data = $this->collect_data();
        return rest_ensure_response( array(
            'generated_at' => current_time('mysql'),
            'source'       => get_bloginfo('name'),
            'version'      => self::VERSION,
            'data'         => $data,
        ) );
    }

    private function get_export_paths(){
        $upload_dir = wp_upload_dir();
        $dir = trailingslashit($upload_dir['basedir']). self::UPLOAD_SUBDIR;
        wp_mkdir_p($dir);
        return array(
            'dir'      => $dir,
            'json'     => $dir.'/export.json',
            'xml'      => $dir.'/export.xml',
            'json_url' => trailingslashit($upload_dir['baseurl']). self::UPLOAD_SUBDIR . '/export.json',
            'xml_url'  => trailingslashit($upload_dir['baseurl']). self::UPLOAD_SUBDIR . '/export.xml',
        );
    }

    public function build_exports(){
        $data  = $this->collect_data();
        $paths = $this->get_export_paths();

        @file_put_contents($paths['json'], wp_json_encode(array(
            'generated_at'=> current_time('mysql'),
            'site'        => array( 'name'=>get_bloginfo('name'), 'url'=>home_url('/'), 'language'=>get_bloginfo('language') ),
            'version'     => self::VERSION,
            'data'        => $data,
        ), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));

        $xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<zohiSeoExport version="'.esc_attr(self::VERSION).'" generatedAt="'.esc_attr(current_time('mysql')).'">';
        $xml .= '<site><name>'.esc_html(get_bloginfo('name')).'</name><url>'.esc_url(home_url('/')).'</url><language>'.esc_html(get_bloginfo('language')).'</language></site>';
        $xml .= '<items>';
        foreach ( $data['items'] as $item ) {
            $xml .= '<item>';
            foreach ( $item as $key=>$val ) {
                if ( is_array($val) ) {
                    $xml .= '<'.sanitize_key($key).'>';
                    foreach ( $val as $k2=>$v2 ) {
                        if ( is_array($v2) ) {
                            $xml .= '<node>';
                            foreach ( $v2 as $k3=>$v3 ) {
                                $xml .= '<'.sanitize_key($k3).'>'.esc_html( is_scalar($v3)?$v3:wp_json_encode($v3) ).'</'.sanitize_key($k3).'>';
                            }
                            $xml .= '</node>';
                        } else {
                            $xml .= '<'.sanitize_key(is_string($k2)?$k2:'value').'>'.esc_html($v2).'</'.sanitize_key(is_string($k2)?$k2:'value').'>';
                        }
                    }
                    $xml .= '</'.sanitize_key($key).'>';
                } else {
                    $xml .= '<'.sanitize_key($key).'>'.esc_html($val).'</'.sanitize_key($key).'>';
                }
            }
            $xml .= '</item>';
        }
        $xml .= '</items>';
        if ( ! empty($data['robots']) ) $xml .= '<robots><![CDATA['.$data['robots'].']]></robots>';
        if ( ! empty($data['sitemaps']) ) {
            $xml .= '<sitemaps>';
            foreach ( $data['sitemaps'] as $sm ) $xml .= '<sitemap>'.esc_url($sm).'</sitemap>';
            $xml .= '</sitemaps>';
        }
        $xml .= '</zohiSeoExport>';
        @file_put_contents($paths['xml'], $xml);

        // Klasör listelemeyi engelle (Apache için basit)
        if ( file_exists($paths['dir']) ) {
            @file_put_contents($paths['dir'].'/index.html', '');
            @file_put_contents($paths['dir'].'/.htaccess', "Options -Indexes\n<Files ~ \"\\.(json|xml)$\">\n  Require all granted\n</Files>\n" );
        }
    }

    private function detect_seo_plugin() {
        $a = array();
        if ( defined('WPSEO_VERSION') )    $a[] = 'Yoast SEO';
        if ( defined('RANK_MATH_VERSION') )$a[] = 'Rank Math';
        if ( defined('AIOSEO_VERSION') )   $a[] = 'All in One SEO';
        if ( defined('SEOPRESS_VERSION') ) $a[] = 'SEOPress';
        return $a;
    }

    private function collect_data(){
        $opts = get_option(self::OPTION_KEY);
        $post_types = isset($opts['post_types']) ? (array)$opts['post_types'] : array('post','page');
        $items = array();

        $q = new WP_Query(array(
            'post_type'      => $post_types,
            'post_status'    => 'publish',
            'posts_per_page' => 500,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
            'fields'         => 'ids',
        ));

        foreach ( $q->posts as $post_id ) {
            $meta = get_post_meta($post_id);
            $permalink = get_permalink($post_id);
            $title = get_the_title($post_id);

            $seo = array(
                'title'       => isset($meta['_yoast_wpseo_title'][0]) ? $meta['_yoast_wpseo_title'][0] : ( isset($meta['rank_math_title'][0]) ? $meta['rank_math_title'][0] : ( isset($meta['_aioseo_title'][0]) ? $meta['_aioseo_title'][0] : '' ) ),
                'description' => isset($meta['_yoast_wpseo_metadesc'][0]) ? $meta['_yoast_wpseo_metadesc'][0] : ( isset($meta['rank_math_description'][0]) ? $meta['rank_math_description'][0] : ( isset($meta['_aioseo_description'][0]) ? $meta['_aioseo_description'][0] : '' ) ),
                'focus_key'   => isset($meta['_yoast_wpseo_focuskw'][0]) ? $meta['_yoast_wpseo_focuskw'][0] : ( isset($meta['rank_math_focus_keyword'][0]) ? $meta['rank_math_focus_keyword'][0] : '' ),
                'canonical'   => isset($meta['_yoast_wpseo_canonical'][0]) ? $meta['_yoast_wpseo_canonical'][0] : ( isset($meta['rank_math_canonical_url'][0]) ? $meta['rank_math_canonical_url'][0] : '' ),
                'og_title'    => isset($meta['_yoast_wpseo_opengraph-title'][0]) ? $meta['_yoast_wpseo_opengraph-title'][0] : '',
                'og_desc'     => isset($meta['_yoast_wpseo_opengraph-description'][0]) ? $meta['_yoast_wpseo_opengraph-description'][0] : '',
                'og_image'    => isset($meta['_yoast_wpseo_opengraph-image'][0]) ? $meta['_yoast_wpseo_opengraph-image'][0] : '',
                'tw_title'    => isset($meta['_yoast_wpseo_twitter-title'][0]) ? $meta['_yoast_wpseo_twitter-title'][0] : '',
                'tw_desc'     => isset($meta['_yoast_wpseo_twitter-description'][0]) ? $meta['_yoast_wpseo_twitter-description'][0] : '',
            );

            if ( $seo['title'] === '' )       $seo['title'] = $title;
            if ( $seo['description'] === '' ) $seo['description'] = wp_trim_words( wp_strip_all_tags( get_post_field('post_content', $post_id) ), 28 );

            $terms_data = array();
            if ( ! empty($opts['include_tax']) ) {
                $taxes = get_object_taxonomies( get_post_type($post_id), 'objects' );
                foreach ( $taxes as $tax=>$obj ) {
                    $terms = wp_get_post_terms( $post_id, $tax, array( 'fields'=>'all' ) );
                    if ( !is_wp_error($terms) && !empty($terms) ) {
                        $arr = array();
                        foreach ( $terms as $t ) {
                            $arr[] = array(
                                'term_id' => $t->term_id,
                                'slug'    => $t->slug,
                                'name'    => $t->name,
                                'link'    => get_term_link( $t ),
                            );
                        }
                        $terms_data[$tax] = $arr;
                    }
                }
            }

            $thumb = get_the_post_thumbnail_url( $post_id, 'full' );
            $content_plain = wp_strip_all_tags( get_post_field('post_content', $post_id) );
            $wc = str_word_count( $content_plain );

            $items[] = array(
                'id'          => $post_id,
                'type'        => get_post_type( $post_id ),
                'title'       => $title,
                'permalink'   => $permalink,
                'seo'         => $seo,
                'featured'    => $thumb ? $thumb : '',
                'published'   => get_post_time('c', true, $post_id),
                'modified'    => get_post_modified_time('c', true, $post_id),
                'word_count'  => $wc,
                'reading_min' => max( 1, ceil( $wc / 200 ) ),
                'terms'       => $terms_data,
            );
        }

        // robots.txt
        $robots_content = '';
        $robots_file = ABSPATH . 'robots.txt';
        if ( file_exists( $robots_file ) ) {
            $robots_content = (string) @file_get_contents( $robots_file );
        } else {
            ob_start(); do_action('do_robots'); $robots_content = ob_get_clean();
        }

        // sitemaps
        $sitemaps = array();
        $home = home_url('/');
        foreach ( array('sitemap.xml','sitemap_index.xml','wp-sitemap.xml') as $p ) {
            $sitemaps[] = trailingslashit($home) . ltrim($p,'/');
        }

        return array(
            'detected_seo_plugins' => !empty($opts['detect_seo_plugins']) ? $this->detect_seo_plugin() : array(),
            'items'   => $items,
            'robots'  => $robots_content,
            'sitemaps'=> array_values(array_unique($sitemaps)),
        );
    }
}

new Zohi_AI_SEO();

endif;
