<?php
/*
Plugin Name: Zohi Pop-up
Plugin URI: https://www.zohi.net
Description: Ziyaretçilere görsel tabanlı pop-up göstermenizi sağlayan basit bir eklenti.
Version: 1.0.1
Author: Zohi Bilişim
Author URI: https://www.zohi.net
License: GPL2
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'ZOHI_POPUP_VERSION', '1.0.1' );
define( 'ZOHI_POPUP_PLUGIN_FILE', __FILE__ );

class Zohi_Popup_Plugin {

    public function __construct() {
        add_action( 'init', array( $this, 'register_post_type' ) );
        add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
        add_action( 'save_post_zohi_popup', array( $this, 'save_meta_boxes' ), 10, 2 );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
        add_action( 'wp_head', array( $this, 'frontend_styles' ) );
        add_action( 'wp_footer', array( $this, 'render_popup' ) );
    }

    public function register_post_type() {
        $labels = array(
            'name'               => 'Zohi Pop-up',
            'singular_name'      => 'Pop-up',
            'menu_name'          => 'Zohi Pop-up',
            'name_admin_bar'     => 'Zohi Pop-up',
            'add_new'            => 'Yeni Pop-up Ekle',
            'add_new_item'       => 'Yeni Pop-up Ekle',
            'new_item'           => 'Yeni Pop-up',
            'edit_item'          => 'Pop-up Düzenle',
            'view_item'          => 'Pop-up Görüntüle',
            'all_items'          => 'Tüm Pop-uplar',
            'search_items'       => 'Pop-up Ara',
            'not_found'          => 'Pop-up bulunamadı',
            'not_found_in_trash' => 'Çöpte pop-up bulunamadı',
        );

        $args = array(
            'labels'             => $labels,
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'            => true,
            'show_in_menu'       => true,
            'query_var'          => false,
            'rewrite'            => false,
            'capability_type'    => 'post',
            'has_archive'        => false,
            'hierarchical'       => false,
            'menu_position'      => 25,
            'menu_icon'          => 'dashicons-format-image',
            'supports'           => array( 'title' ),
        );

        register_post_type( 'zohi_popup', $args );
    }

    public function add_meta_boxes() {
        add_meta_box(
            'zohi_popup_settings',
            'Pop-up Ayarları',
            array( $this, 'render_settings_metabox' ),
            'zohi_popup',
            'normal',
            'high'
        );
    }

    public function render_settings_metabox( $post ) {
        wp_nonce_field( 'zohi_popup_save_meta', 'zohi_popup_meta_nonce' );

        $image_id   = get_post_meta( $post->ID, '_zohi_popup_image_id', true );
        $image_url  = '';
        if ( $image_id ) {
            $img = wp_get_attachment_image_src( $image_id, 'medium' );
            if ( $img ) {
                $image_url = $img[0];
            }
        }

        $target_url = get_post_meta( $post->ID, '_zohi_popup_target_url', true );
        $display_on = get_post_meta( $post->ID, '_zohi_popup_display_on', true );
        $delay      = get_post_meta( $post->ID, '_zohi_popup_delay', true );
        $show_mode  = get_post_meta( $post->ID, '_zohi_popup_show_mode', true );

        if ( empty( $delay ) && $delay !== '0' ) {
            $delay = 2;
        }

        if ( empty( $show_mode ) ) {
            $show_mode = 'once';
        }

        $is_home_checked  = ( $display_on === 'home' || $display_on === 'both' || empty( $display_on ) );
        $is_inner_checked = ( $display_on === 'inner' || $display_on === 'both' || empty( $display_on ) );
        ?>
        <table class="form-table">
            <tr>
                <th scope="row"><label for="zohi_popup_image">Pop-up Görseli</label></th>
                <td>
                    <div id="zohi-popup-image-wrapper">
                        <?php if ( $image_url ) : ?>
                            <img src="<?php echo esc_url( $image_url ); ?>" alt="" style="max-width: 200px; height: auto;">
                        <?php endif; ?>
                    </div>
                    <input type="hidden" id="zohi_popup_image_id" name="zohi_popup_image_id" value="<?php echo esc_attr( $image_id ); ?>">
                    <button type="button" class="button" id="zohi-popup-upload-button">Görsel Seç / Yükle</button>
                    <button type="button" class="button" id="zohi-popup-remove-button" <?php echo $image_id ? '' : 'style="display:none;"'; ?>>Görseli Kaldır</button>
                </td>
            </tr>
            <tr>
                <th scope="row"><label for="zohi_popup_target_url">Pop-up Linki</label></th>
                <td>
                    <input type="text" id="zohi_popup_target_url" name="zohi_popup_target_url" value="<?php echo esc_attr( $target_url ); ?>" class="regular-text" placeholder="https://">
                    <p class="description">Pop-up görsele tıklandığında nereye girmesini isterseniz o linki yazınız.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Hangi Sayfalarda Gösterilsin?</th>
                <td>
                    <label>
                        <input type="checkbox" name="zohi_popup_display_home" value="1" <?php checked( $is_home_checked ); ?>>
                        Anasayfa
                    </label><br>
                    <label>
                        <input type="checkbox" name="zohi_popup_display_inner" value="1" <?php checked( $is_inner_checked ); ?>>
                        İç sayfalar
                    </label>
                    <p class="description">Pop-up'ın hangi sayfalarda görüntüleneceğini seçiniz.</p>
                </td>
            </tr>
            <tr>
                <th scope="row"><label for="zohi_popup_delay">Kaç saniye sonra açılsın?</label></th>
                <td>
                    <input type="number" min="0" id="zohi_popup_delay" name="zohi_popup_delay" value="<?php echo esc_attr( $delay ); ?>" class="small-text">
                    <span>saniye</span>
                    <p class="description">Varsayılan olarak 2 saniye sonra açılır.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Gösterim Şekli</th>
                <td>
                    <label>
                        <input type="radio" name="zohi_popup_show_mode" value="once" <?php checked( $show_mode, 'once' ); ?>>
                        Bir kez göster (çerez ile)
                    </label><br>
                    <label>
                        <input type="radio" name="zohi_popup_show_mode" value="always" <?php checked( $show_mode, 'always' ); ?>>
                        Her sayfada göster
                    </label>
                    <p class="description">Bir kez göster seçildiğinde, ziyaretçinin tarayıcısında çerez oluşturulur ve pop-up tekrar gösterilmez.</p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function save_meta_boxes( $post_id, $post ) {
        if ( ! isset( $_POST['zohi_popup_meta_nonce'] ) || ! wp_verify_nonce( $_POST['zohi_popup_meta_nonce'], 'zohi_popup_save_meta' ) ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Image ID
        $image_id = isset( $_POST['zohi_popup_image_id'] ) ? intval( $_POST['zohi_popup_image_id'] ) : 0;
        if ( $image_id ) {
            update_post_meta( $post_id, '_zohi_popup_image_id', $image_id );
        } else {
            delete_post_meta( $post_id, '_zohi_popup_image_id' );
        }

        // Target URL
        $target_url = isset( $_POST['zohi_popup_target_url'] ) ? esc_url_raw( $_POST['zohi_popup_target_url'] ) : '';
        if ( $target_url ) {
            update_post_meta( $post_id, '_zohi_popup_target_url', $target_url );
        } else {
            delete_post_meta( $post_id, '_zohi_popup_target_url' );
        }

        // Display on
        $display_home  = isset( $_POST['zohi_popup_display_home'] );
        $display_inner = isset( $_POST['zohi_popup_display_inner'] );

        if ( $display_home && $display_inner ) {
            $display_on = 'both';
        } elseif ( $display_home ) {
            $display_on = 'home';
        } elseif ( $display_inner ) {
            $display_on = 'inner';
        } else {
            $display_on = '';
        }

        if ( $display_on ) {
            update_post_meta( $post_id, '_zohi_popup_display_on', $display_on );
        } else {
            delete_post_meta( $post_id, '_zohi_popup_display_on' );
        }

        // Delay
        $delay = isset( $_POST['zohi_popup_delay'] ) ? intval( $_POST['zohi_popup_delay'] ) : 2;
        if ( $delay < 0 ) {
            $delay = 0;
        }
        update_post_meta( $post_id, '_zohi_popup_delay', $delay );

        // Show mode
        $show_mode = isset( $_POST['zohi_popup_show_mode'] ) ? sanitize_text_field( $_POST['zohi_popup_show_mode'] ) : 'once';
        if ( ! in_array( $show_mode, array( 'once', 'always' ), true ) ) {
            $show_mode = 'once';
        }
        update_post_meta( $post_id, '_zohi_popup_show_mode', $show_mode );
    }

    public function admin_scripts( $hook ) {
        global $post;
        if ( $hook === 'post-new.php' || $hook === 'post.php' ) {
            if ( isset( $post ) && $post->post_type === 'zohi_popup' ) {
                wp_enqueue_media();
                wp_enqueue_script(
                    'zohi-popup-admin',
                    plugin_dir_url( ZOHI_POPUP_PLUGIN_FILE ) . 'zohi-popup-admin.js',
                    array( 'jquery' ),
                    ZOHI_POPUP_VERSION,
                    true
                );
            }
        }
    }

    public function frontend_styles() {
        if ( is_admin() ) {
            return;
        }
        ?>
        <style type="text/css">
            .zohi-popup-overlay {
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0,0,0,0.6);
                display: none;
                align-items: center;
                justify-content: center;
                z-index: 9999;
            }
            .zohi-popup-content {
                position: relative;
                max-width: 90%;
                max-height: 90%;
                background: #ffffff;
                padding: 10px;
                box-shadow: 0 0 20px rgba(0,0,0,0.4);
            }
            .zohi-popup-content img {
                max-width: 100%;
                height: auto;
                display: block;
            }
            .zohi-popup-close {
                position: absolute;
                top: 5px;
                right: 5px;
                width: 28px;
                height: 28px;
                border-radius: 50%;
                border: none;
                background: #000000;
                color: #ffffff;
                cursor: pointer;
                font-size: 18px;
                line-height: 1;
            }
            .zohi-popup-close:hover {
                opacity: 0.85;
            }
        </style>
        <?php
    }

    public function render_popup() {
        if ( is_admin() ) {
            return;
        }

        $context = 'inner';
        if ( is_front_page() || is_home() ) {
            $context = 'home';
        }

        $args = array(
            'post_type'      => 'zohi_popup',
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'meta_query'     => array(
                'relation' => 'OR',
                array(
                    'key'   => '_zohi_popup_display_on',
                    'value' => 'both',
                ),
                array(
                    'key'   => '_zohi_popup_display_on',
                    'value' => $context,
                ),
            ),
        );

        $popup_query = new WP_Query( $args );

        if ( ! $popup_query->have_posts() ) {
            wp_reset_postdata();
            return;
        }

        $popup_query->the_post();
        $post_id   = get_the_ID();
        $image_id  = get_post_meta( $post_id, '_zohi_popup_image_id', true );
        $image_url = '';

        if ( $image_id ) {
            $img = wp_get_attachment_image_src( $image_id, 'full' );
            if ( $img ) {
                $image_url = $img[0];
            }
        }

        if ( ! $image_url ) {
            wp_reset_postdata();
            return;
        }

        $target_url = get_post_meta( $post_id, '_zohi_popup_target_url', true );
        $delay      = get_post_meta( $post_id, '_zohi_popup_delay', true );
        if ( $delay === '' ) {
            $delay = 2;
        }
        $delay_ms = intval( $delay ) * 1000;
        if ( $delay_ms < 0 ) {
            $delay_ms = 0;
        }

        $show_mode = get_post_meta( $post_id, '_zohi_popup_show_mode', true );
        if ( empty( $show_mode ) ) {
            $show_mode = 'once';
        }

        ?>
        <div class="zohi-popup-overlay" id="zohi-popup-overlay" aria-hidden="true">
            <div class="zohi-popup-content" role="dialog" aria-modal="true">
                <button class="zohi-popup-close" type="button" aria-label="Kapat">&times;</button>
                <?php if ( $target_url ) : ?>
                    <a href="<?php echo esc_url( $target_url ); ?>">
                        <img src="<?php echo esc_url( $image_url ); ?>" alt="Zohi Pop-up">
                    </a>
                <?php else : ?>
                    <img src="<?php echo esc_url( $image_url ); ?>" alt="Zohi Pop-up">
                <?php endif; ?>
            </div>
        </div>
        <script type="text/javascript">
            document.addEventListener('DOMContentLoaded', function () {
                var overlay = document.getElementById('zohi-popup-overlay');
                if (!overlay) return;
                var delay = <?php echo $delay_ms; ?>;
                var showMode = "<?php echo esc_js( $show_mode ); ?>";
                var popupId = "<?php echo esc_js( $post_id ); ?>";

                function getCookie(name) {
                    var value = "; " + document.cookie;
                    var parts = value.split("; " + name + "=");
                    if (parts.length === 2) return parts.pop().split(";").shift();
                }

                function setCookie(name, value, days) {
                    var d = new Date();
                    d.setTime(d.getTime() + (days*24*60*60*1000));
                    var expires = "expires=" + d.toUTCString();
                    document.cookie = name + "=" + value + ";" + expires + ";path=/";
                }

                if (showMode === 'once') {
                    var cookieName = "zohi_popup_shown_" + popupId;
                    if (getCookie(cookieName)) {
                        return; // daha önce gösterilmiş, tekrar gösterme
                    }
                }

                setTimeout(function () {
                    overlay.style.display = 'flex';
                    if (showMode === 'once') {
                        var cookieName = "zohi_popup_shown_" + popupId;
                        setCookie(cookieName, "1", 30); // 30 gün sakla
                    }
                }, delay);

                overlay.addEventListener('click', function (e) {
                    if (e.target === overlay || e.target.classList.contains('zohi-popup-close')) {
                        overlay.style.display = 'none';
                    }
                });
            });
        </script>
        <?php

        wp_reset_postdata();
    }
}

new Zohi_Popup_Plugin();
